﻿using System;
using System.Collections.Generic;
using System.ComponentModel;
using System.Data;
using System.Drawing;
using System.Linq;
using System.Text;
using System.Windows.Forms;
using RevisionAnalyser.Projects;
using RevisionAnalyser.Global;
using System.IO;
using System.Threading;
using System.Diagnostics;
using RevisionAnalyser.Controls;
using System.Runtime.InteropServices;

namespace RevisionAnalyser
{
    public partial class MainForm : LoadableForm
    {
        [DllImport("user32.dll")]
        private static extern IntPtr GetForegroundWindow();

        // Hot keys
        private const int HK_NEW = 300201;
        private const int HK_OPEN = 300202;
        private const int HK_REV_PREV = 300203;
        private const int HK_REV_NEXT = 300204;
        private const int HK_REV_LVL1 = 300205;
        private const int HK_REV_LVL2 = 300206;
        private const int HK_REV_LVL3 = 300207;
        private const int HK_REV_LVL4 = 300208;
        private const int HK_REV_LVL5 = 300209;
        private const int HK_REV_LVL6 = 300210;
        private const int HK_REV_LVL7 = 300211;
        private const int HK_REV_LVL8 = 300212;
        private const int HK_REV_LVL9 = 300213;
        private const int HK_COLLAPSE_ALL = 300214;
        private const int HK_EXPAND_ALL = 300215;

        public MainForm()
        {
            InitializeComponent();
        }

        public void UpdateSolidSX(string path)
        {
            ShowLoadScreen("Updating dataset...");
            uxSolidSX.OpenSolidSX(path);
        }

        public void OpenSolidSX(string path)
        {
            uxSolidSX.OpenSolidSX(path);
        }

        private void NewProject()
        {
            NewProjectForm frmNewProject = new NewProjectForm();
            if (frmNewProject.ShowDialog() == DialogResult.OK)
            {
                OpenProject(frmNewProject.GetProjectFile(), true);
            }
        }

        private void OpenProject(string file, bool firstRun)
        {
            if (CurrentProject.Instance.Project != null)
            {
                CloseProject();
            }
            ShowLoadScreen("Opening project...");
            try
            {
                uxSolidSX.OpenSolidSX(Constants.EMPTY_DB_PATH);
                CurrentProject.Instance.Open(this, file);
                if (firstRun)
                {
                    CurrentProject.Instance.Project.FirstRun();
                }
                CurrentProject.Instance.Project.InitMenu(uxProjectMenu);
                CurrentProject.Instance.Project.Opened();
                Text = String.Format("{0} - {1}", Constants.APP_TITLE, file);
                uxProjectMenu.Visible = true;
                uxCloseProjectMenu.Enabled = true;
                uxExpandAllMenu.Enabled = true;
                uxCollapseAllMenu.Enabled = true;
                uxExpandLevelMenu.Enabled = true;
                uxNodeFilterMenu.Enabled = true;
                uxRevisionSlider.Visible = true;
                uxSolidSX.Visible = true;
                uxRevisionSlider.Focus();
            }
            catch (Exception ex)
            {
                TopMostMessageBox.Show("Could not open the project file. Do you have SolidSX installed? \n\nError message: " + ex.Message, "Could not open project", MessageBoxButtons.OK, MessageBoxIcon.Error);
                HideLoadScreen();
                CloseProject();
            }
        }

        private void OpenProject(bool firstRun)
        {
            if (uxOpenDialog.ShowDialog() == DialogResult.OK)
            {
                OpenProject(uxOpenDialog.FileName, false);
            }
        }

        private void CloseProject()
        {
            try
            {
                CurrentProject.Instance.Close();
                Text = Constants.APP_TITLE;
                uxProjectMenu.DropDownItems.Clear();
                uxCloseProjectMenu.Enabled = false;
                uxProjectMenu.Visible = false;
                uxExpandAllMenu.Enabled = false;
                uxCollapseAllMenu.Enabled = false;
                uxExpandLevelMenu.Enabled = false;
                uxNodeFilterMenu.Enabled = false;
                uxSolidSX.Visible = false;
                uxSolidSX.CloseSolidSX();
                uxRevisionSlider.Visible = false;
                uxRevisionSet.ClearRevisions();
            }
            catch (Exception ex)
            {
                TopMostMessageBox.Show("Could not close the project file.\n\nError message: " + ex.Message, "Could not close project", MessageBoxButtons.OK, MessageBoxIcon.Error);
            }
        }

        private void OpenOptions()
        {
            OptionsForm frmOptions = new OptionsForm();
            frmOptions.ShowDialog();
        }

        private void OpenAbout()
        {
            AboutForm frmAbout = new AboutForm();
            frmAbout.ShowDialog();
        }

        private void OpenHelp()
        {
            try
            {
                Process.Start(Application.StartupPath + "\\User Manual.pdf");
            }
            catch
            {
                MessageBox.Show("Could not open the 'User Manual.pdf' file.", "Error", MessageBoxButtons.OK, MessageBoxIcon.Warning);
            }
        }

        public RevisionSet GetRevisionSet()
        {
            return uxRevisionSet;
        }

        public SolidSXControl GetSolidSXControl()
        {
            return uxSolidSX;
        }

        private void MainForm_Load(object sender, EventArgs e)
        {
            AppInit();
        }

        private void uxNewProjectMenu_Click(object sender, EventArgs e)
        {
            NewProject();
        }

        private void uxAboutMenu_Click(object sender, EventArgs e)
        {
            OpenAbout();
        }

        private void uxExitMenu_Click(object sender, EventArgs e)
        {
            Close();
        }

        private void uxOpenProjectMenu_Click(object sender, EventArgs e)
        {
            OpenProject(false);
        }

        private void uxCloseProjectMenu_Click(object sender, EventArgs e)
        {
            CloseProject();
        }

        private void MainForm_FormClosing(object sender, FormClosingEventArgs e)
        {
            CloseSplash();
            base.OnClosing(e);
        }

        private void uxOptionsMenu_Click(object sender, EventArgs e)
        {
            OpenOptions();
        }

        private void uxViewHelpMenu_Click(object sender, EventArgs e)
        {
            OpenHelp();
        }

        private void MainForm_ResizeEnd(object sender, EventArgs e)
        {
            uxRevisionSlider.Update();
        }

        private void uxRevisionSlider_RevisionMouseLeave(object sender, RevisionSlider.RevisionEventArgs e)
        {
            uxStatusLabel.Text = "Ready";
        }

        private void uxDebugMenu_Click(object sender, EventArgs e)
        {
            // Debug code here
        }

        private void uxExpandAllMenu_Click(object sender, EventArgs e)
        {
            uxSolidSX.ExpandAll();
        }

        private void uxCollapseAllMenu_Click(object sender, EventArgs e)
        {
            uxSolidSX.CollapseAll();
        }

        private void uxExpandLevelMenu_Click(object sender, EventArgs e)
        {
            uxSolidSX.CollapseAll();
            uxSolidSX.ExpandLevel(Byte.Parse(((ToolStripMenuItem)sender).Tag.ToString()));
        }

        private void uxNodeFilterMenu_Click(object sender, EventArgs e)
        {
            uxSolidSX.SetNodeInclusionFilter("numberOfRevisions", Byte.Parse(((ToolStripMenuItem)sender).Tag.ToString()));
        }

        private void uxRevisionSlider_RevisionSelected(object sender, RevisionSlider.RevisionEventArgs e)
        {
            uxSolidSX.SelectRevision(e.Revision);
        }

        protected override void WndProc(ref Message m)
        {
            base.WndProc(ref m);
            if (m.Msg == HotKeyRegister.WM_HOTKEY)
            {
                if (GetForegroundWindow() != Handle)
                {
                    return;
                }
                switch ((int)m.WParam)
                {
                    case HK_NEW:
                        uxNewProjectMenu_Click(null, null);
                        break;
                    case HK_OPEN:
                        uxOpenProjectMenu_Click(null, null);
                        break;
                    case HK_REV_PREV:
                        break;
                    case HK_REV_NEXT:
                        break;
                    case HK_REV_LVL1:
                        uxExpandLevelMenu_Click(uxExpandLevel1Menu, null);
                        break;
                    case HK_REV_LVL2:
                        uxExpandLevelMenu_Click(uxExpandLevel2Menu, null);
                        break;
                    case HK_REV_LVL3:
                        uxExpandLevelMenu_Click(uxExpandLevel3Menu, null);
                        break;
                    case HK_REV_LVL4:
                        uxExpandLevelMenu_Click(uxExpandLevel4Menu, null);
                        break;
                    case HK_REV_LVL5:
                        uxExpandLevelMenu_Click(uxExpandLevel5Menu, null);
                        break;
                    case HK_REV_LVL6:
                        uxExpandLevelMenu_Click(uxExpandLevel6Menu, null);
                        break;
                    case HK_REV_LVL7:
                        uxExpandLevelMenu_Click(uxExpandLevel7Menu, null);
                        break;
                    case HK_REV_LVL8:
                        uxExpandLevelMenu_Click(uxExpandLevel8Menu, null);
                        break;
                    case HK_REV_LVL9:
                        uxExpandLevelMenu_Click(uxExpandLevel9Menu, null);
                        break;
                    case HK_COLLAPSE_ALL:
                        uxCollapseAllMenu_Click(null, null);
                        break;
                    case HK_EXPAND_ALL:
                        uxExpandAllMenu_Click(null, null);
                        break;
                }
            }
        }

        private void uxSolidSX_Loaded(object sender, EventArgs e)
        {
            HideLoadScreen();
        }

        private void uxNodeFilterClear_Click(object sender, EventArgs e)
        {
            uxSolidSX.ShowRevisionFilter(uxRevisionSet.GetRevisionList());
        }
    }
}